<?php
require_once __DIR__ . '/../config/database.php';

function jsonResponse($success, $message = '', $data = null, $httpCode = 200) {
    http_response_code($httpCode);
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode([
        'success' => (bool)$success,
        'message' => $message,
        'data'    => $data
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

function getInputJSON() {
    $raw = file_get_contents('php://input');
    $data = json_decode($raw, true);
    return is_array($data) ? $data : [];
}

function sanitize($value) {
    return htmlspecialchars(trim((string)$value), ENT_QUOTES, 'UTF-8');
}

function generateToken(): string {
    return bin2hex(random_bytes(32));
}

function getBearerToken(): ?string {
    $headers = function_exists('getallheaders') ? getallheaders() : [];
    $auth = $headers['Authorization'] ?? $headers['authorization'] ?? '';
    if ($auth && preg_match('/Bearer\s+(.*)$/i', $auth, $m)) return trim($m[1]);
    if (!empty($_GET['token'])) return trim($_GET['token']);
    return null;
}

function authUser(PDO $pdo): array {
    $token = getBearerToken();
    if (!$token) jsonResponse(false, 'Missing auth token', null, 401);
    $stmt = $pdo->prepare("SELECT u.* FROM user_auth_tokens t JOIN users u ON u.id = t.user_id WHERE t.token = ? AND t.expires_at > NOW()");
    $stmt->execute([$token]);
    $user = $stmt->fetch();
    if (!$user) jsonResponse(false, 'Invalid or expired token', null, 401);
    return $user;
}

function getAppSetting(PDO $pdo, string $key, $default = null) {
    $stmt = $pdo->prepare("SELECT setting_value FROM app_settings WHERE setting_key = ?");
    $stmt->execute([$key]);
    $row = $stmt->fetch();
    return $row ? $row['setting_value'] : $default;
}

function setAppSetting(PDO $pdo, string $key, string $value): void {
    $stmt = $pdo->prepare("INSERT INTO app_settings (setting_key, setting_value) VALUES (?, ?) ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value)");
    $stmt->execute([$key, $value]);
}

function userTotalPoints(PDO $pdo, int $userId): int {
    $stmt = $pdo->prepare("SELECT COALESCE(SUM(points),0) AS total FROM user_points WHERE user_id = ?");
    $stmt->execute([$userId]);
    $row = $stmt->fetch();
    return (int)($row['total'] ?? 0);
}

function addUserPoints(PDO $pdo, int $userId, int $points, string $reason = null): void {
    $stmt = $pdo->prepare("INSERT INTO user_points (user_id, points, reason) VALUES (?, ?, ?)");
    $stmt->execute([$userId, $points, $reason]);
}

function refundPoints(PDO $pdo, int $userId, int $points): void {
    addUserPoints($pdo, $userId, $points, 'Refund (payout rejected)');
}

function issueAuthToken(PDO $pdo, int $userId, string $ttl = '+24 hours'): array {
    $token = generateToken();
    $expiresAt = (new DateTime($ttl))->format('Y-m-d H:i:s');
    $pdo->prepare("INSERT INTO user_auth_tokens (user_id, token, expires_at) VALUES (?, ?, ?)")
        ->execute([$userId, $token, $expiresAt]);
    return ['token' => $token, 'expires_at' => $expiresAt];
}