<?php
require_once __DIR__ . '/../../core/functions.php';
$pdo = getDB();
$method = $_SERVER['REQUEST_METHOD'];
$route  = isset($_GET['__route']) ? trim($_GET['__route'], '/') : '';

// POST /api/v1/users/register
if ($method === 'POST' && $route === 'register') {
    $in = getInputJSON();
    $name = sanitize($in['name'] ?? '');
    $email = filter_var($in['email'] ?? '', FILTER_VALIDATE_EMAIL);
    $password = (string)($in['password'] ?? '');
    if (!$name || !$email || strlen($password) < 6) jsonResponse(false, 'Invalid input');
    $hash = password_hash($password, PASSWORD_DEFAULT);
    try {
        $pdo->prepare("INSERT INTO users (name,email,password_hash) VALUES (?,?,?)")->execute([$name,$email,$hash]);
        jsonResponse(true, 'Registration successful');
    } catch (PDOException $e) {
        if ($e->getCode()==='23000') jsonResponse(false,'Email already exists');
        jsonResponse(false,'Registration failed');
    }
}

// POST /api/v1/users/login
if ($method === 'POST' && $route === 'login') {
    $in = getInputJSON();
    $email = filter_var($in['email'] ?? '', FILTER_VALIDATE_EMAIL);
    $password = (string)($in['password'] ?? '');
    if (!$email || !$password) jsonResponse(false, 'Invalid input');
    $stmt = $pdo->prepare("SELECT * FROM users WHERE email=? LIMIT 1");
    $stmt->execute([$email]);
    $user = $stmt->fetch();
    if (!$user || !password_verify($password, $user['password_hash'])) jsonResponse(false,'Invalid credentials',null,401);
    $tok = issueAuthToken($pdo, (int)$user['id']);
    jsonResponse(true,'Login successful',[
        'token'=>$tok['token'],
        'expires_at'=>$tok['expires_at'],
        'user'=>[
            'id'=>(int)$user['id'],
            'name'=>$user['name'],
            'is_pro_subscriber'=>(bool)$user['is_pro_subscriber'],
            'pro_expiry_date'=>$user['pro_expiry_date']
        ]
    ]);
}

// GET /api/v1/users/profile (Protected)
if ($method === 'GET' && $route === 'profile') {
    $user = authUser($pdo);
    $totalPoints = userTotalPoints($pdo, (int)$user['id']);
    jsonResponse(true,'OK',[
        'id'=>(int)$user['id'],
        'name'=>$user['name'],
        'email'=>$user['email'],
        'is_pro_subscriber'=>(bool)$user['is_pro_subscriber'],
        'pro_expiry_date'=>$user['pro_expiry_date'],
        'total_points'=>$totalPoints
    ]);
}

// POST /api/v1/users/reset_password_request
if ($method === 'POST' && $route === 'reset_password_request') {
    $in = getInputJSON();
    $email = filter_var($in['email'] ?? '', FILTER_VALIDATE_EMAIL);
    if (!$email) jsonResponse(false, 'Invalid email');
    $stmt = $pdo->prepare("SELECT id FROM users WHERE email=? LIMIT 1");
    $stmt->execute([$email]);
    $u = $stmt->fetch();
    if (!$u) jsonResponse(true, 'If that email exists, a reset will be sent'); // do not reveal existence
    $token = generateToken();
    $expiresAt = (new DateTime('+30 minutes'))->format('Y-m-d H:i:s');
    $pdo->prepare("INSERT INTO password_reset_tokens (user_id, token, expires_at) VALUES (?,?,?)")
        ->execute([(int)$u['id'],$token,$expiresAt]);
    // TODO: send email/SMS with token or link
    jsonResponse(true,'Password reset token created',[ 'reset_token'=>$token, 'expires_at'=>$expiresAt ]);
}

// POST /api/v1/users/reset_password
if ($method === 'POST' && $route === 'reset_password') {
    $in = getInputJSON();
    $token = (string)($in['token'] ?? '');
    $new = (string)($in['new_password'] ?? '');
    if (strlen($new) < 6) jsonResponse(false,'Password too short');
    $stmt = $pdo->prepare("SELECT user_id FROM password_reset_tokens WHERE token=? AND expires_at>NOW() LIMIT 1");
    $stmt->execute([$token]);
    $row = $stmt->fetch();
    if (!$row) jsonResponse(false,'Invalid or expired token');
    $hash = password_hash($new, PASSWORD_DEFAULT);
    $pdo->beginTransaction();
    try {
        $pdo->prepare("UPDATE users SET password_hash=? WHERE id=?")->execute([$hash,(int)$row['user_id']]);
        $pdo->prepare("DELETE FROM password_reset_tokens WHERE user_id=?")->execute([(int)$row['user_id']]);
        $pdo->commit();
        jsonResponse(true,'Password updated');
    } catch (Exception $e) {
        $pdo->rollBack();
        jsonResponse(false,'Failed to update password');
    }
}

jsonResponse(false,'Route not found',null,404);