<?php
require_once __DIR__ . '/../core/functions.php';
require_once __DIR__ . '/includes/auth_check.php';
include __DIR__ . '/includes/header.php';
$pdo = getDB();

// Fetch lessons to allow content assignment
$lessons = $pdo->query("SELECT id, lesson_title_bn FROM lessons ORDER BY sort_order")->fetchAll();

// Handle form submissions for both creating and updating content items
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  $lessonId = (int)($_POST['lesson_id'] ?? 0);
  $itemType = $_POST['item_type'] ?? '';
  $textItalian = sanitize($_POST['text_italian'] ?? '');
  $textBengali = sanitize($_POST['text_bengali'] ?? '');
  $pronunciation = sanitize($_POST['pronunciation_bengali'] ?? '');
  $contextNote = sanitize($_POST['context_note'] ?? '');
  $audioUrl = sanitize($_POST['audio_url'] ?? '');
  $imageUrl = sanitize($_POST['image_url'] ?? '');

  if (!$lessonId || !$itemType || !$textItalian || !$textBengali) {
    echo '<div class="alert alert-danger">Invalid input.</div>';
  } else {
    if (!empty($_POST['content_id'])) {
      // Update existing content item
      $contentId = (int)$_POST['content_id'];
      $stmt = $pdo->prepare("UPDATE content_items SET item_type = ?, text_italian = ?, text_bengali = ?, pronunciation_bengali = ?, context_note = ?, audio_url = ?, image_url = ?, lesson_id = ? WHERE id = ?");
      $stmt->execute([$itemType, $textItalian, $textBengali, $pronunciation, $contextNote, $audioUrl, $imageUrl, $lessonId, $contentId]);
      echo '<div class="alert alert-success">Content item updated.</div>';
    } else {
      // Create new content item
      $stmt = $pdo->prepare("INSERT INTO content_items (item_type, text_italian, text_bengali, pronunciation_bengali, context_note, audio_url, image_url, lesson_id) VALUES (?, ?, ?, ?, ?, ?, ?, ?)");
      $stmt->execute([$itemType, $textItalian, $textBengali, $pronunciation, $contextNote, $audioUrl, $imageUrl, $lessonId]);
      echo '<div class="alert alert-success">Content item created.</div>';
    }
  }
}

// Handle content item deletion
if (isset($_GET['delete'])) {
  $contentId = (int)$_GET['delete'];
  $stmt = $pdo->prepare("DELETE FROM content_items WHERE id = ?");
  $stmt->execute([$contentId]);
  echo '<div class="alert alert-danger">Content item deleted.</div>';
}

// Fetch all content items with associated lessons
$contentItems = $pdo->query("SELECT ci.id, ci.item_type, ci.text_italian, ci.text_bengali, l.lesson_title_bn FROM content_items ci JOIN lessons l ON ci.lesson_id = l.id ORDER BY l.sort_order, ci.id")->fetchAll();

// If editing a content item, load the current data
if (isset($_GET['edit'])) {
    $contentId = (int)$_GET['edit'];
    $stmt = $pdo->prepare("SELECT * FROM content_items WHERE id = ? LIMIT 1");
    $stmt->execute([$contentId]);
    $contentItem = $stmt->fetch();
    if (!$contentItem) {
        echo '<div class="alert alert-danger">Content item not found.</div>';
    } else {
        $itemType = $contentItem['item_type'];
        $textItalian = $contentItem['text_italian'];
        $textBengali = $contentItem['text_bengali'];
        $pronunciation = $contentItem['pronunciation_bengali'];
        $contextNote = $contentItem['context_note'];
        $audioUrl = $contentItem['audio_url'];
        $imageUrl = $contentItem['image_url'];
        $lessonId = $contentItem['lesson_id'];
    }
} else {
    // Default values for creating a new content item
    $itemType = '';
    $textItalian = '';
    $textBengali = '';
    $pronunciation = '';
    $contextNote = '';
    $audioUrl = '';
    $imageUrl = '';
    $lessonId = 0;
}
?>
<h3 class="mb-3"><?php echo isset($_GET['edit']) ? 'Edit Content Item' : 'Create Content Item'; ?></h3>

<!-- Content Item Form: For both creating and editing -->
<form method="post" class="mb-3">
  <?php if (isset($contentItem)): ?>
    <input type="hidden" name="content_id" value="<?php echo $contentItem['id']; ?>">
  <?php endif; ?>
  <div class="mb-3">
    <label class="form-label">Item Type</label>
    <select name="item_type" class="form-control" required>
      <option value="WORD" <?php echo $itemType === 'WORD' ? 'selected' : ''; ?>>WORD</option>
      <option value="PHRASE" <?php echo $itemType === 'PHRASE' ? 'selected' : ''; ?>>PHRASE</option>
    </select>
  </div>
  <div class="mb-3">
    <label class="form-label">Lesson</label>
    <select name="lesson_id" class="form-control" required>
      <?php foreach ($lessons as $lesson): ?>
        <option value="<?php echo $lesson['id']; ?>" <?php echo $lesson['id'] === $lessonId ? 'selected' : ''; ?>>
          <?php echo htmlspecialchars($lesson['lesson_title_bn']); ?>
        </option>
      <?php endforeach; ?>
    </select>
  </div>
  <div class="mb-3">
    <label class="form-label">Italian Text</label>
    <input type="text" name="text_italian" class="form-control" value="<?php echo htmlspecialchars($textItalian); ?>" required>
  </div>
  <div class="mb-3">
    <label class="form-label">Bengali Translation</label>
    <input type="text" name="text_bengali" class="form-control" value="<?php echo htmlspecialchars($textBengali); ?>" required>
  </div>
  <div class="mb-3">
    <label class="form-label">Pronunciation (Bengali)</label>
    <input type="text" name="pronunciation_bengali" class="form-control" value="<?php echo htmlspecialchars($pronunciation); ?>">
  </div>
  <div class="mb-3">
    <label class="form-label">Context Note</label>
    <textarea name="context_note" class="form-control"><?php echo htmlspecialchars($contextNote); ?></textarea>
  </div>
  <div class="mb-3">
    <label class="form-label">Audio URL</label>
    <input type="text" name="audio_url" class="form-control" value="<?php echo htmlspecialchars($audioUrl); ?>">
  </div>
  <div class="mb-3">
    <label class="form-label">Image URL</label>
    <input type="text" name="image_url" class="form-control" value="<?php echo htmlspecialchars($imageUrl); ?>">
  </div>
  <button class="btn btn-primary"><?php echo isset($contentItem) ? 'Save Content Item' : 'Create Content Item'; ?></button>
</form>

<!-- Content Item List -->
<table class="table table-striped">
  <thead>
    <tr><th>ID</th><th>Item Type</th><th>Italian Text</th><th>Bengali Translation</th><th>Lesson</th><th>Actions</th></tr>
  </thead>
  <tbody>
    <?php foreach ($contentItems as $contentItem): ?>
      <tr>
        <td><?php echo $contentItem['id']; ?></td>
        <td><?php echo $contentItem['item_type']; ?></td>
        <td><?php echo htmlspecialchars($contentItem['text_italian']); ?></td>
        <td><?php echo htmlspecialchars($contentItem['text_bengali']); ?></td>
        <td><?php echo htmlspecialchars($contentItem['lesson_title_bn']); ?></td>
        <td>
          <a class="btn btn-sm btn-outline-info" href="/admin/content.php?edit=<?php echo $contentItem['id']; ?>">Edit</a>
          <a class="btn btn-sm btn-outline-danger" href="/admin/content.php?delete=<?php echo $contentItem['id']; ?>">Delete</a>
        </td>
      </tr>
    <?php endforeach; ?>
  </tbody>
</table>
<?php include __DIR__ . '/includes/footer.php'; ?>
